/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 **************************************************************************************************
 * @file Application.cpp
 *
 * @brief This file contains some common functions which might be used by any example application. 
 * 
 * @author Elmar
 *
 */

#include "stdafx.h"
#include "Debug.h"
#include "Common.h"
#include "Application.h"


/*************************************************************************************************
 * @brief This function sends a packet and waits for the confirmation.
 * Packets which will come up meanwhile will be passed to the standard packet handler.
 * 
 * @param ptApp     Pointer to application data.
 * @param ptSendPkt Packet which will be send.
 * @param ptRecvPkt Packet which the function has received.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT SendRecvPkt(APPLICATION_T* ptApp, CIFX_PACKET* ptSendPkt, CIFX_PACKET* ptRecvPkt)
{
  TLR_RESULT tResult = TLR_S_OK; 

  /* fire the packet */
  tResult = xChannelPutPacket(ptApp->tCommon.hChannel, ptSendPkt, ptApp->tCommon.ulTimeout);
  if (TLR_S_OK != tResult)
  {
    return tResult;
  }

  /* ok, at this point we have successfully sent a packet */

  /* check for packets to receive */
  /* the ulCnfCmd is always: ulReqCmd | 0x01 */
  TLR_UINT32 ulCnfCmd = ptSendPkt->tHeader.ulCmd | 0x01;  
  
  while(CIFX_NO_ERROR == (tResult = xChannelGetPacket(ptApp->tCommon.hChannel, sizeof(*ptRecvPkt), ptRecvPkt, ptApp->tCommon.ulTimeout)))
  {
    /* check for our packet */
    if(ptRecvPkt->tHeader.ulCmd == ulCnfCmd)
    {
      /* it is our packet, so return its status as result */
      tResult = ptRecvPkt->tHeader.ulState;
      
      /* Note: we also return the packet which we have received (by reference, see signature of function) */

      /* we have received our packet, so we can break here*/
      break;
    }
    else
    {
      /* it is something else, so place it in the application packet handler */
      HandlePacket(ptApp, ptRecvPkt);
    }
  }

  return tResult;
}

/*************************************************************************************************
 * @brief This function sends and receives an empty packet.
 * 
 * @param ptApp Pointer to application data.
 * @param ulCmd Command code which will be inserted into packet header.
 *
 * @return TLR_RESULT Returns TLR_S_OK if no error, otherwise it will return an error code.
 *
 */
TLR_RESULT SendRecvEmptyPkt(APPLICATION_T* ptApp, TLR_UINT32 ulCmd)
{
  
  CIFX_PACKET tSendPkt = {{0}};
  CIFX_PACKET tRecvPkt = {{0}};

  TLR_RESULT tResult = TLR_S_OK;

  /* all register packets are empty packet, */
  /* which means that they contain no data, */
  /* so a cast to a specific type is not necessary */
  TLR_EMPTY_PACKET_T* ptReq = (TLR_EMPTY_PACKET_T*) &tSendPkt;

  /* ulSrc and ulDest are set by GetPacket / SendPacket */
  ptReq->tHead.ulCmd    = ulCmd;
  ptReq->tHead.ulDest   = 0x20;
  ptReq->tHead.ulDestId = 0;
  ptReq->tHead.ulExt    = 0;
  ptReq->tHead.ulId     = 0;
  ptReq->tHead.ulLen    = 0;
  ptReq->tHead.ulRout   = 0;
  ptReq->tHead.ulSrc    = 0;
  ptReq->tHead.ulSrcId  = 0;
  ptReq->tHead.ulSta    = 0;

  tResult = SendRecvPkt(ptApp, &tSendPkt, &tRecvPkt);
  return tResult;
}


